import { supabase } from '@/config/supabase';
import {
  Notification,
  NotificationStatus,
  NotificationType,
  DeliveryChannel,
  SendNotificationRequest
} from './types';

export const notificationsService = {
  // List notifications for user (with filters)
  async listNotifications(user_id: string, filters: any) {
    let query = supabase.from('notifications').select('*').eq('user_id', user_id).is('deleted_at', null);
    if (filters.status) query = query.eq('status', filters.status);
    if (filters.type) query = query.eq('type', filters.type);
    if (filters.date_range) {
      const [from, to] = filters.date_range;
      query = query.gte('created_at', from).lte('created_at', to);
    }
    const { data, error } = await query.order('created_at', { ascending: false });
    return { data, error };
  },

  // Mark as read
  async markAsRead(user_id: string, id: string) {
    const { data, error } = await supabase.from('notifications').update({ status: 'read', read: true }).eq('id', id).eq('user_id', user_id).select().single();
    return { data, error };
  },

  // Soft delete
  async deleteNotification(user_id: string, id: string) {
    const { data, error } = await supabase.from('notifications').update({ deleted_at: new Date().toISOString() }).eq('id', id).eq('user_id', user_id).select().single();
    return { data, error };
  },

  // Send notification (admin or system)
  async sendNotification(req: SendNotificationRequest) {
    // 1. Insert notification
    const { data, error } = await supabase.from('notifications').insert({
      user_id: req.user_id,
      vendor_id: req.vendor_id,
      type: req.type,
      title: req.title,
      message: req.message,
      status: 'unread',
      delivery_channel: req.send_push ? 'push' : 'in_app',
      read: false,
      sent_at: new Date().toISOString(),
      created_at: new Date().toISOString()
    }).select().single();
    if (error) return { error };
    // 2. If send_push, trigger push delivery (placeholder)
    if (req.send_push && req.user_id) {
      // Fetch push token from user_sessions
      const { data: session } = await supabase.from('user_sessions').select('push_token').eq('user_id', req.user_id).single();
      if (session?.push_token) {
        // Call push service (placeholder)
        // await sendPushNotification(session.push_token, req.title, req.message);
      }
    }
    return { data };
  }
}; 